<?php

namespace App\Http\Requests;

use App\Rules\Mobile;
use App\Rules\Password;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;
use Illuminate\Validation\Rule;

class MemberRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules(): array
    {
        if ($this->getMethod() == 'POST') {
            return [
                'first_name' => [
                    'required'
                    , 'max:50'
                ],
                'last_name' => [
                    'required'
                    , 'max:50'
                ],
                'email' => [
                    'required',
                    'email',
                    Rule::unique('users')->where('deleted', 0)
                ],
                'phone' => [
                    'required', new Mobile()
                    , Rule::unique('users')->where('deleted', 0)
                ],
                'password' => [
                    'required'
                    , new Password()
                ]
            ];
        }
        if ($this->getMethod() == 'PATCH') {
            return [
                'first_name' => [
                    'sometimes',
                    'required'
                    , 'max:50'
                ],
                'last_name' => [
                    'sometimes',
                    'required'
                    , 'max:50'
                ],
                'email' => [
                    'sometimes',
                    'required',
                    'email',
                    Rule::unique('users')->where('deleted', 0)->ignore(Auth::user()->id)
                ],
                'phone' => [
                    'sometimes',
                    'required', new Mobile()
                    , Rule::unique('users')->where('deleted', 0)->ignore(Auth::user()->id)
                ],
                'password' => [
                    'nullable'
                    , new Password()
                ],
                'image' => [
                    'nullable'
                ],
                'city_id' => [
                    'sometimes',
                    'required'
                    , Rule::exists('world_cities', 'id')
                ],
                'gender' => [
                    'nullable'
                    , 'in:male,female'
                ],
                'locale' => [
                    'sometimes',
                    'required',
                    'in:en,ar'
                ],
                'birth_date' => [
                    'nullable'
                    , 'date_format:Y-m-d'
                ],
                'firebase_token' => [
                    'nullable'
                ],
                'orders' => [
                    'sometimes',
                    'required',
                    'boolean'
                ],
                'promotions' => [
                    'sometimes',
                    'required',
                    'boolean'
                ],
                'others' => [
                    'sometimes',
                    'required',
                    'boolean'
                ],
                'emails' => [
                    'sometimes',
                    'required',
                    'boolean'
                ],
                'image_quality' => [
                    'sometimes',
                    'required',
                    Rule::in(['High Quality', 'Normal Quality'])
                ]
            ];
        }
        return [
            'first_name' => [
                'required'
                , 'max:50'
            ],
            'last_name' => [
                'required'
                , 'max:50'
            ],
            'email' => [
                'required',
                'email',
                Rule::unique('users')->where('deleted', 0)->ignore(Auth::user()->id)
            ],
            'phone' => [
                'required', new Mobile()
                , Rule::unique('users')->where('deleted', 0)->ignore(Auth::user()->id)
            ],
            'password' => [
                'nullable'
                , new Password()
            ],
            'image' => [
                'nullable'
            ],
            'city_id' => [
                'required'
                , Rule::exists('world_cities', 'id')
            ],
            'gender' => [
                'nullable'
                , 'in:male,female'
            ],
            'locale' => [
                'required',
                'in:en,ar'
            ],
            'birth_date' => [
                'nullable'
                , 'date_format:Y-m-d'
            ],
            'firebase_token' => [
                'nullable'
            ],
            'orders' => [
                'required',
                'boolean'
            ],
            'promotions' => [
                'required',
                'boolean'
            ],
            'others' => [
                'required',
                'boolean'
            ],
            'emails' => [
                'required',
                'boolean'
            ],
            'image_quality' => [
                'required',
                Rule::in(['High Quality', 'Normal Quality'])
            ]
        ];
    }

    protected function failedValidation(Validator $validator)
    {
        throw new HttpResponseException(Response::failure($validator->errors()->first()));
    }
}
