<?php

namespace App\Exceptions;

use Illuminate\Auth\AuthenticationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Str;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Throwable;

class Handler extends ExceptionHandler
{
    /**
     * A list of exception types with their corresponding custom log levels.
     *
     * @var array<class-string<\Throwable>, \Psr\Log\LogLevel::*>
     */
    protected $levels = [
        //
    ];

    /**
     * A list of the exception types that are not reported.
     *
     * @var array<int, class-string<\Throwable>>
     */
    protected $dontReport = [
        //
    ];

    /**
     * A list of the inputs that are never flashed to the session on validation exceptions.
     *
     * @var array<int, string>
     */
    protected $dontFlash = [
        'current_password',
        'password',
        'password_confirmation',
    ];

    /**
     * Register the exception handling callbacks for the application.
     *
     * @return void
     */
    public function register()
    {
        $this->reportable(function (Throwable $e) {
            //
        });
    }

    public function render($request, Throwable $e)
    {
        if ($e instanceof ModelNotFoundException) {
            return Response::failure(__('common.message_no_data'));
        }
        if ($e instanceof AuthenticationException) {
            return Response::success([
                'session' => 'expired',
                'error' => 'Session has expired. Please sign in and try again.'
            ], 302);
        }
        if ($e instanceof NotFoundHttpException) {
            return Response::success([
                'error' => 'The requested file does not exists'
            ], 404);
        }
        if (Str::contains($request->getRequestUri(), 'admin')
            || Str::contains($request->getRequestUri(), 'api')
            || Str::contains($request->getRequestUri(), 'delivery')) {
            $this->logError($request, $e, 'api_error');
        }
        $this->logError($request, $e, 'error');

        return Response::failure($e->getMessage() . '' . $e->getLine(), 500);
        //return parent::render($request, $e);
    }

    private function logError(Request $request, Throwable $e, $channel)
    {
        try {
            $requestBody = json_encode($request->all());
            $userIDorIP = strval(optional($request->user())->id ?: $request->ip());
            $exceptionClass = get_class($e);
            Log::channel($channel)->error("
                ############################ Exception ############################\n
                Exception Class:\n
                $exceptionClass\n
                --------------------------------------------------------------------------------\n\n
                Request:\n
                $requestBody\n
                --------------------------------------------------------------------------------\n\n
                Request URL:\n
                {$request->fullUrl()}\n
                --------------------------------------------------------------------------------\n\n
                Request User or IP:\n
                {$userIDorIP}\n
                --------------------------------------------------------------------------------\n\n
                Message:\n
                {$e->getMessage()}\n
                --------------------------------------------------------------------------------\n
                Line:\n
                {$e->getLine()}\n
                --------------------------------------------------------------------------------\n
                File:\n
                {$e->getFile()}\n
                --------------------------------------------------------------------------------\n
                Trace:\n
                {$e->getTraceAsString()}\n
                --------------------------------------------------------------------------------\n
                ############################ Exception ############################\n\n

                ");
        } catch (Throwable $ignore) {

        }
    }
}
