<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Http\Requests\CheckoutRequest;
use App\Models\Order;
use App\Providers\OrderPaymentFailed;
use App\Providers\OrderPlaced;
use App\Services\Checkout\CheckoutService;
use App\Services\NotificationService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;

class OrderController extends Controller
{
    private $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
        $this->middleware('auth:api')->except(['postProcess']);
        $this->middleware('is_member')->except(['postProcess']);
        $this->middleware('is_active')->except(['postProcess']);
    }

    public function index(): JsonResponse
    {
        return Response::success(Order::query()
            ->where('pruned', '=', '0')
            ->where('user_id', '=', Auth::user()->id)
            ->orderByDesc('date')
            ->select(['id', 'date', 'number', 'state', 'total', 'shipping_address_id', 'seller_id'])
        );
    }

    public function show($id): JsonResponse
    {
        return Response::success(Order::query()
            ->where('user_id', '=', Auth::user()->id)
            ->with([
                'user'
                , 'delivery'
                , 'coupon'
                , 'billingAddress'
                , 'shippingAddress'
                , 'seller'
                , 'payment'
                , 'deliveryOption'
                , 'orderItems.product'
                , 'address'
            ])
            ->find($id)
        );
    }

    public function checkout(CheckoutRequest $request, CheckoutService $checkoutService): JsonResponse
    {
        $order = $checkoutService->checkout($request->validated(), $request->query('config') === 'as_draft');
        if ($order->post_process == '0') {
            OrderPlaced::dispatch($order);
        }
        return Response::success($order);
    }

    public function postProcess(Request $request, CheckoutService $checkoutService): JsonResponse
    {
        $result = $checkoutService->postProcess($request);
        if ($result['status'] === 'success' && $result['order'] != null) {
            OrderPlaced::dispatch($result['order']);
        } else {
            OrderPaymentFailed::dispatch($result['order'], null, null, null);
            return Response::success(null);
        }
        return Response::success(null);
    }
}
