<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;

class ProductController extends Controller
{
    public function __construct()
    {
        if (config('app.middlewares_enabled') === true) {
            $this->middleware('auth:api')->only(['favorites', 'toggleProductLike']);
            $this->middleware('is_member')->only(['favorites', 'toggleProductLike']);
            $this->middleware('is_active')->only(['favorites', 'toggleProductLike']);
        }
    }


    public function todayDeals(): JsonResponse
    {
        $user = Auth::guard('api')->user();
        return Response::success(
            Product::search(request()->query('word_filter'))
                ->with(['outOfStock'])
                ->whereNotNull('special_offer')
                ->when($user != null, function ($query) use ($user) {
                    $query->leftJoin('product_user', function ($join) use ($user) {
                        $join->on('products.id', '=', 'product_user.product_id')
                            ->where('product_user.user_id', '=', $user->id)
                            ->whereNull('products.deleted_at');
                    })->select('products.*', DB::raw("(product_user.id IS NOT NULL) in_favorite"));
                })
                ->get()
                ->where('active_special_offer', '=', true)
                ->take(is_int(request()->query('limit', 10)) ? request()->query('limit', 10) : 10)->values()->all()
        );
    }

    public function topRating(): JsonResponse
    {
        $user = Auth::guard('api')->user();
        return Response::success(
            Product::search(request()->query('word_filter'))
                ->with(['outOfStock'])
                ->join('reviews', function ($join) {
                    $join->on('reviews.product_id', '=', 'products.id')->where('reviews.state', '=', 'accepted')->whereNull('reviews.deleted_at');
                })
                ->when($user != null, function ($query) use ($user) {
                    $query->leftJoin('product_user', function ($join) use ($user) {
                        $join->on('products.id', '=', 'product_user.product_id')
                            ->where('product_user.user_id', '=', $user->id)
                            ->whereNull('products.deleted_at');
                    })->select('products.*', DB::raw("(product_user.id IS NOT NULL) in_favorite"));
                })
                ->select('products.*', DB::raw('AVG(rating) as average_rating'))
                ->groupBy('id')
                ->orderByDesc('average_rating')
                ->take(is_int(request()->query('limit', 10)) ? request()->query('limit', 10) : 10)
        );
    }

    public function index(): JsonResponse
    {
        $user = Auth::guard('api')->user();
        return Response::success(
            Product::search(request()->query('word_filter'))
                ->with(['outOfStock'])
                ->when($user != null, function ($query) use ($user) {
                    $query->leftJoin('product_user', function ($join) use ($user) {
                        $join->on('products.id', '=', 'product_user.product_id')
                            ->where('product_user.user_id', '=', $user->id)
                            ->whereNull('products.deleted_at');
                    })->select('products.*', DB::raw("(product_user.id IS NOT NULL) in_favorite"));
                })
                ->withCount(['reviews as average_rating' => function ($query) {
                    $query->where('state', '=', 'accepted')->whereNull('deleted_at')->select(DB::raw('avg(rating)'));
                }])
                ->codeFilter(request()->query('code_filter'))
                ->brandFilter(request()->query('brand_filter'))
                ->sellerFilter(request()->query('seller_filter'))
                ->categoriesFilter(request()->query('categories_filter'))
                ->latestProducts(request()->query('latest_products'))
                ->active(true)
                ->bestCollection(request()->query('best_collection'))
                ->newlyAdded(request()->query('newly_added'))
                ->flashDeals(request()->query('flash_deals'))
                ->selected(request()->query('selected'))
        );
    }

    public function show($id): JsonResponse
    {
        $user = Auth::guard('api')->user();
        //DB::enableQueryLog();
        return Response::success(
            Product::query()
                ->when($user != null, function ($query) use ($user) {
                    $query->leftJoin('product_user', function ($join) use ($user) {
                        $join->on('products.id', '=', 'product_user.product_id')
                            ->where('product_user.user_id', '=', $user->id)
                            ->whereNull('products.deleted_at');
                    })->select('products.*', DB::raw("(product_user.id IS NOT NULL) in_favorite"));
                })
                ->withCount(['reviews as average_rating' => function ($query) {
                    $query->where('state', '=', 'accepted')->whereNull('deleted_at')->select(DB::raw('avg(rating)'));
                }])
                ->with([
                    'brand'
                    , 'seller'
                    , 'associations.products:id,name_en,name_ar,price,image'
                    , 'categories'
                    , 'outOfStock'
                    , 'availableDeliveryOptions'
                    , 'availablePayments'
                    , 'colors'
                    , 'sizes'
                    , 'attributes.group'
                    , 'reviews' => function ($query) {
                        $query->with(['user:id,first_name,last_name,email,phone,image'])->where('state', '=', 'accepted');
                    }
                ])->find($id)
        );
        //return DB::getQueryLog();
    }

    public function favorites(): JsonResponse
    {
        $user = Auth::user();

        return Response::success(
            Product::search(request()->query('word_filter'))
                ->with(['outOfStock'])
                ->join('product_user', function ($join) use ($user) {
                    $join->on('products.id', '=', 'product_user.product_id')
                        ->where('product_user.user_id', '=', $user->id)
                        ->whereNull('products.deleted_at');
                })
                ->withCount(['reviews as average_rating' => function ($query) {
                    $query->where('state', '=', 'accepted')->whereNull('deleted_at')->select(DB::raw('avg(rating)'));
                }])
                ->codeFilter(request()->query('code_filter'))
                ->brandFilter(request()->query('brand_filter'))
                ->sellerFilter(request()->query('seller_filter'))
                ->categoriesFilter(request()->query('categories_filter'))
                ->active(true)
        );
    }

    public function toggleProductLike($id): JsonResponse
    {
        $user = Auth::user();
        if ($user->favoriteProducts()->where('products.id', '=', $id)->first() == null) {
            $message = __('common.message_added_to_favorites');
        } else {
            $message = __('common.message_removed_from_favorites');
        }
        $user->favoriteProducts()->toggle($id);
        return Response::success(['message' => $message]);
    }
}
