<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\CouponRequest;
use App\Models\Coupon;
use App\Models\Order;
use App\Services\CouponService;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;

class CouponController extends Controller
{
    private $couponService;

    public function __construct(CouponService $couponService)
    {
        $this->couponService = $couponService;

        if (config('app.middlewares_enabled') === true) {
            $this->middleware('auth:api');
            $this->middleware('has_permission:view_coupon')->only(['index', 'export', 'show']);
            $this->middleware('has_permission:create_coupon')->only(['store']);
            $this->middleware('has_permission:delete_coupon')->only(['delete']);
        }
    }

    public function index(Request $request): JsonResponse
    {
        $discounts =
            $this->baseQuery($request)
                ->get();
        $arr = [];
        foreach ($discounts->groupBy('code') as $key => $value) {
            $output = [
                'code' => $key,
                'code_group' => '',
                'for_marketing' => '',
                'on_delivery_cost' => '',
                'value' => '',
                'expiration' => '',
                'active' => '',
                'body' => '',
                'title' => ''
            ];
            if (count($value) > 0) {
                $output = array_merge($output, [
                    'for_marketing' => $value[0]['for_marketing'],
                    'code_group' => $value[0]['code_group'],
                    'on_delivery_cost' => $value[0]['on_delivery_cost'],
                    'value' => $value[0]['percentage'] == null ?
                        number_format($value[0]['amount'], 2)
                        : '%' . number_format($value[0]['percentage'], 2),
                    'expiration' => $value[0]['expiration'],
                    'active' => $value[0]['active'],
                    'body' => $value[0]['body'],
                    'title' => $value[0]['title']
                ]);
            }
            array_push($arr, $output);
        }
        return Response::success($arr);
    }

    private function baseQuery(Request $request)
    {
        return Coupon::search($request->query('word_filter'))
            ->when($request->query('for_marketing') != null, function ($query) use ($request) {
                $query->where('for_marketing', '=', $request->query('for_marketing') == 'true' ? '1' : '0');
            })
            ->when($request->query('on_delivery_cost') != null, function ($query) use ($request) {
                $query->where('on_delivery_cost', '=', $request->query('on_delivery_cost') == 'true' ? '1' : '0');
            })
            ->when($request->query('expired_filter') == 'true', function ($query) use ($request) {
                $query->where(function ($query) use ($request) {
                    $query->whereDate('expiration', '<', Carbon::now()->toDateTimeString());
                });
            })
            ->when($request->query('expired_filter') == 'false', function ($query) use ($request) {
                $query->where(function ($query) use ($request) {
                    $query->whereDate('expiration', '>=', Carbon::now()->toDateTimeString());
                });
            })
            ->orderByDesc('creation');
    }

    public function show($id): JsonResponse
    {
        return Response::success(
            Coupon::query()->find($id)
        );
    }

    public function store(CouponRequest $request): JsonResponse
    {
        $data = $request->validated();
        $coupons = $this->couponService->create(Auth::user()->timezone, $data);
        return Response::success($coupons);

    }

    public function update(CouponRequest $request): JsonResponse
    {
        if ($request->input('active') == '1') {
            if (Coupon::query()
                    ->where('code', '=', $request->input('code'))
                    ->where('active', '=', 1)
                    ->count() > 0) {
                return response()->json(['error' => __('common.error_active_coupon_exists')], 400);
            }
        }
        Coupon::query()->where('code_group', '=', $request->input('code_group'))->update(['active' => $request->input('active')]);
        return response()->json(null);
    }

    public function destroy($group): JsonResponse
    {
        Coupon::query()->where('code_group', '=', $group)->delete();
        Coupon::onlyTrashed()->whereNotIn('id', Order::query()->whereNotNull('coupon_id')->pluck('coupon_id'))->forceDelete();
        return Response::success(null);
    }
}
