<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\BestCollectionRequest;
use App\Http\Requests\FlashDealRequest;
use App\Http\Requests\NewlyAddedRequest;
use App\Http\Requests\ProductPatchRequest;
use App\Http\Requests\ProductRequest;
use App\Http\Requests\SelectedRequest;
use App\Models\Payment;
use App\Models\Product;
use App\Services\ProductService;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;

class ProductController extends Controller
{
    private $productService;

    public function __construct(ProductService $productService)
    {
        $this->productService = $productService;
        if (config('app.middlewares_enabled') === true) {
            $this->middleware('auth:api');
            $this->middleware('has_permission:view_product')->only(['index', 'export', 'show']);
            $this->middleware('has_permission:create_product')->only(['store']);
            $this->middleware('has_permission:update_product')->only(['update']);
            $this->middleware('has_permission:delete_product')->only(['delete']);
        }
    }

    public function index(): JsonResponse
    {
        return Response::success(
            Product::search(request()->query('word_filter'))
                ->codeFilter(request()->query('code_filter'))
                ->bestCollection(request()->query('best_collection'))
                ->newlyAdded(request()->query('newly_added'))
                ->flashDeals(request()->query('flash_deal'))
                ->flashDeals(request()->query('flash_deals'))
                ->active(request()->query('activated'))
                ->brandFilter(request()->query('brand_filter'))
                ->categoriesFilter(request()->query('categories_filter'))
                ->selected(request()->query('selected'))
        );
    }

    public function show($id): JsonResponse
    {
        return Response::success(
            Product::query()->with([
                'brand'
                , 'seller'
                , 'associations.products:id,name_en,name_ar,price'
                , 'categories'
                , 'outOfStock'
                , 'availableDeliveryOptions'
                , 'availablePayments'
                , 'colors'
                , 'sizes'
                , 'attributes.group'
            ])->find($id)
        );
    }

    public function store(ProductRequest $request): JsonResponse
    {
        DB::beginTransaction();
        $product = $this->productService->create($request->safe()->all());
        DB::commit();
        return Response::success($product->load([
            'brand'
            , 'seller'
            , 'associations.products:id,name_en,name_ar,price'
            , 'categories'
            , 'outOfStock'
            , 'availableDeliveryOptions'
            , 'availablePayments'
            , 'colors'
            , 'sizes'
            , 'attributes.group'
        ]));
    }

    public function updateFlashDeals(FlashDealRequest $request, $id): JsonResponse
    {
        DB::beginTransaction();
        $product = Product::query()->findOrFail($id);
        $product->update($request->validated());
        DB::commit();
        return Response::success($product->load([
            'brand'
            , 'seller'
            , 'associations.products:id,name_en,name_ar,price'
            , 'categories'
            , 'outOfStock'
            , 'availableDeliveryOptions'
            , 'availablePayments'
            , 'colors'
            , 'sizes'
            , 'attributes.group'
        ]));
    }

    public function updateBestCollection(BestCollectionRequest $request, $id): JsonResponse
    {
        DB::beginTransaction();
        $product = Product::query()->findOrFail($id);
        $product->update($request->validated());
        DB::commit();
        return Response::success($product->load([
            'brand'
            , 'seller'
            , 'associations.products:id,name_en,name_ar,price'
            , 'categories'
            , 'outOfStock'
            , 'availableDeliveryOptions'
            , 'availablePayments'
            , 'colors'
            , 'sizes'
            , 'attributes.group'
        ]));
    }

    public function updateNewlyAdded(NewlyAddedRequest $request, $id): JsonResponse
    {
        DB::beginTransaction();
        $product = Product::query()->findOrFail($id);
        $product->update($request->validated());
        DB::commit();
        return Response::success($product->load([
            'brand'
            , 'seller'
            , 'associations.products:id,name_en,name_ar,price'
            , 'categories'
            , 'outOfStock'
            , 'availableDeliveryOptions'
            , 'availablePayments'
            , 'colors'
            , 'sizes'
            , 'attributes.group'
        ]));
    }

    public function updateSelected(SelectedRequest $request, $id): JsonResponse
    {
        DB::beginTransaction();
        $product = Product::query()->findOrFail($id);
        $product->update($request->validated());
        DB::commit();
        return Response::success($product->load([
            'brand'
            , 'seller'
            , 'associations.products:id,name_en,name_ar,price'
            , 'categories'
            , 'outOfStock'
            , 'availableDeliveryOptions'
            , 'availablePayments'
            , 'colors'
            , 'sizes'
            , 'attributes.group'
        ]));
    }

    public function update(ProductRequest $request, $id): JsonResponse
    {
        DB::beginTransaction();
        if ($request->getMethod() === 'PATCH') {
            $product = Product::query()->findOrFail($id);
            $product->update($request->validated());
        } else {
            $product = $this->productService->update($request->safe()->all(), $id);
        }
        DB::commit();
        return Response::success($product->load([
            'brand'
            , 'seller'
            , 'associations.products:id,name_en,name_ar,price'
            , 'categories'
            , 'outOfStock'
            , 'availableDeliveryOptions'
            , 'availablePayments'
            , 'colors'
            , 'sizes'
            , 'attributes.group'
        ]));
    }

    public function destroy($id): JsonResponse
    {
        Product::query()->findOrFail($id)->delete();
        return Response::success(null);
    }
}
