<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Response;

class SalesController extends Controller
{

    public function __construct()
    {
        if (config('app.middlewares_enabled') === true) {
            $this->middleware('auth:api');
            $this->middleware('is_admin');
        }
    }

    public function summary(): JsonResponse
    {
        $year = request()->query('year', Carbon::now()->year);
        return Response::success([
            'sales' => Order::query()
                ->whereYear('date', '=', $year)
                ->sum('total'),
            'paid_orders' => Order::query()
                ->whereYear('date', '=', $year)
                ->where('payment_state', '=', 'paid')
                ->sum('total'),
            'customers' => User::query()->roleFilter('Member')->count()
        ]);
    }

    public function salesChart(): JsonResponse
    {
        $year = request()->query('year', Carbon::now()->year);
        $orders = Order::query()
            ->whereYear('date', '=', $year)
            ->get()
            ->groupBy(function ($item) {
                return Carbon::parse($item->date)->format('m');
            });
        $chart = [];
        foreach ($orders as $key => $value) {
            array_push($chart, [
                'month' => $key,
                'sum' => $value->sum('total')
            ]);
        }
        return Response::success(
            $chart
        );
    }

    public function newCustomers(): JsonResponse
    {
        return Response::success(User::query()->roleFilter('Member')->latest()->take(5)->get(['id', 'first_name', 'last_name', 'email']));
    }

    public function newOrders(): JsonResponse
    {
        return Response::success(Order::query()
            ->with(['user:id,email,first_name,last_name'])
            ->withCount(['orderItems'])
            ->latest('date')
            ->take(5)
            ->get(['id', 'total']));
    }
}
