<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SellerRequest;
use App\Models\Seller;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Str;
use Khsing\World\Models\City;

class SellerController extends Controller
{

    public function __construct()
    {
        if (config('app.middlewares_enabled') === true) {
            $this->middleware('auth:api');
            $this->middleware('has_permission:view_seller')->only(['index', 'export', 'show']);
            $this->middleware('has_permission:create_seller')->only(['store']);
            $this->middleware('has_permission:update_seller')->only(['update']);
            $this->middleware('has_permission:delete_seller')->only(['delete']);
        }
    }

    public function index(): JsonResponse
    {
        return Response::success(
            User::search(request()->query('word_filter'))
                ->with(['seller'])
                ->phoneFilter(request()->query('phone_filter'))
                ->active(request()->query('activated'))
                ->seller()
        );
    }

    public function show($id): JsonResponse
    {
        return Response::success(
            User::query()->seller()->with(['city', 'appPreference', 'seller'])->find($id)
        );
    }

    public function store(SellerRequest $request): JsonResponse
    {
        DB::beginTransaction();
        $data = $request->validated();
        $userInput = Arr::only($data, ['first_name', 'last_name', 'phone', 'email', 'image', 'city_id']);
        $userInput['verified'] = $userInput['activated'] = '1';
        $userInput['timezone'] = City::query()->find($request->input('city_id'))->country->timezone;
        $userInput['role'] = 'Seller';
        $userInput['password'] = Str::random(10);
        $user = User::query()->create($userInput);
        $sellerInput = Arr::except($data, ['first_name', 'last_name', 'phone', 'email', 'image', 'city_id', 'activated']);
        $sellerInput['user_id'] = $user->id;
        Seller::query()->create($sellerInput);
        DB::commit();

        return Response::success($user->load(['city', 'appPreference', 'seller']));

    }

    public function update(SellerRequest $request, $id): JsonResponse
    {
        $user = User::query()
            ->seller()
            ->findOrFail($id);
        $data = $request->validated();
        $userInput = Arr::only($data, ['first_name', 'last_name', 'phone', 'email', 'image', 'city_id', 'activated']);
        $userInput['timezone'] = City::query()->find($request->input('city_id'))->country->timezone;
        $user->update($userInput);
        $user->seller()->update(Arr::except($data, ['first_name', 'last_name', 'phone', 'email', 'image', 'city_id', 'activated']));
        return Response::success($user->load(['city', 'appPreference', 'seller']));
    }

    public function destroy($id): JsonResponse
    {
        DB::beginTransaction();
        User::query()->seller()->findOrFail($id)->delete();
        DB::commit();
        return Response::success(null);

    }
}
