<?php

namespace App\Http\Requests;

use App\Rules\Mobile;
use App\Rules\Password;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class UserRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules(): array
    {
        $baseValidation = [
            'first_name' => [
                'required'
                , 'max:50'
            ],
            'last_name' => [
                'required'
                , 'max:50'
            ],
            'image' => [
                'nullable'
            ],
            'city_id' => [
                'required'
                , Rule::exists('world_cities', 'id')
            ],
        ];
        if (Str::contains($this->getRequestUri(), 'manager')) {
            $baseValidation = array_merge($baseValidation, [
                'role_id' => ['required', Rule::exists('roles', 'id')->whereNotIn('id', [1, 2, 3, 4, 5])]
            ]);
        }
        if (Str::contains($this->getRequestUri(), 'seller')) {
            $baseValidation = array_merge($baseValidation, [
                'supplier_since' => ['required', 'date_format:Y-m-d'],
                'corporation_name' => ['required', 'max:50'],
                'longitude' => ['nullable'],
                'latitude' => ['nullable'],
                'description_en' => ['nullable'],
                'description_ar' => ['nullable'],
                'address_en' => ['nullable'],
                'address_ar' => ['nullable'],
            ]);
        }
        if ($this->getMethod() === 'POST') {
            return array_merge($baseValidation, [
                'email' => [
                    'required',
                    'email',
                    Rule::unique('users')->where('deleted', 0)
                ],
                'phone' => [
                    'required', new Mobile()
                    , Rule::unique('users')->where('deleted', 0)
                ],
                'password' => [
                    'required'
                    , new Password()
                ],
                'verified' => [
                    'nullable',
                    'boolean'
                ],

            ]);
        }
        return array_merge($baseValidation, [
            'email' => [
                'required',
                'email',
                'max:100', Rule::unique('users')->where('deleted', 0)->ignore(request()->route('id'))
            ],
            'phone' => ['required'
                , 'max:50', new Mobile()
                , Rule::unique('users')->where('deleted', 0)->ignore(request()->route('id'))
            ],
            'password' => [
                'nullable'
                , new Password()
            ],
            'gender' => [
                'nullable'
                , 'in:male,female'
            ],
            'locale' => [
                'required',
                'in:en,ar'
            ],
            'birth_date' => [
                'nullable'
                , 'date_format:Y-m-d'
            ],
            'activated' => [
                'required'
                , 'boolean'
            ],
            'verified' => [
                'nullable',
                'boolean'
            ],
        ]);
    }

    protected function failedValidation(Validator $validator)
    {
        throw new HttpResponseException(Response::failure($validator->errors()->first()));
    }
}
