<?php

namespace App\Models;

use App\Http\Traits\HasCurrency;
use App\Http\Traits\HasSearch;
use App\Http\Traits\HasUniqueSoftDeletes;
use App\Http\Traits\ProductMedia;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\MediaLibrary\HasMedia;

class Product extends Model implements HasMedia
{
    use HasFactory, SoftDeletes, HasSearch, ProductMedia, HasUniqueSoftDeletes, HasCurrency;

    protected $guarded = ['id'];

    protected $preserveMedia = true;
    protected $searchable = [
        'name_en',
        'name_ar',
        'short_description_en',
        'short_description_ar',
        'description_en',
        'description_ar',
        'tags',
        'code',
        'barcode',
        'sku'
    ];
    protected $casts = [
        'subtract_stock' => 'boolean',
        'requires_shipping' => 'boolean',
        'accepted' => 'boolean',
        'activated' => 'boolean',
        'meta_tag_keywords_en' => 'array',
        'meta_tag_keywords_ar' => 'array',
        'tags' => 'array',
        'seo' => 'array',
        'dimensions' => 'array',
        'image' => 'array',
        'images' => 'array',
        'discount' => 'array',
        'special_offer' => 'array',
        'newly_added' => 'boolean',
        'best_collection' => 'boolean',
        'flash_deal' => 'boolean',
        'selected' => 'boolean'
    ];
    protected $appends = [
        'active_discount',
        'active_special_offer'
    ];

    public function getPriceAttribute($value)
    {
        return $this->convertCurrency($value);
    }

    public function getDiscountAttribute($value)
    {
        if ($value != null) {
            $value = !is_array($value) ? json_decode($value, true) : $value;
            if ($value['price'] != null) {
                $value['price'] = $this->convertCurrency($value['price']);
                return $value;
            }
        }
        return null;
    }

    public function getActiveDiscountAttribute(): bool
    {
        if ($this->discount != null) {
            return Carbon::now($this->timezone)
                ->betweenIncluded(
                    Carbon::parse($this->discount['start_date'], $this->timezone)
                    , Carbon::parse($this->discount['end_date'], $this->timezone));
        }
        return false;
    }

    public function getSpecialOfferAttribute($value)
    {
        if ($value != null) {
            $value = !is_array($value) ? json_decode($value, true) : $value;
            if ($value['price'] != null) {
                $value['price'] = $this->convertCurrency($value['price']);
                return $value;
            }
        }
        return null;
    }

    public function getActiveSpecialOfferAttribute(): bool
    {
        if ($this->special_offer != null) {
            return Carbon::now($this->timezone)
                ->betweenIncluded(
                    Carbon::parse($this->special_offer['start_date'], $this->timezone)
                    , Carbon::parse($this->special_offer['end_date'], $this->timezone));
        }
        return false;
    }

    public function scopeCodeFilter($query, $code)
    {
        return $query->when($code !== null, function ($query) use ($code) {
            $query->where('code', '=', $code);
        });
    }

    public function scopeBrandFilter($query, $brand)
    {
        return $query->when($brand !== null, function ($query) use ($brand) {
            $query->where('brand_id', '=', $brand);
        });
    }

    public function scopeSellerFilter($query, $seller)
    {
        return $query->when($seller !== null, function ($query) use ($seller) {
            $query->where('seller_id', '=', $seller);
        });
    }

    public function scopeCategoriesFilter($query, $categories)
    {
        return $query->when($categories !== null, function ($query) use ($categories) {
            $query->whereHas('categories', function ($query) use ($categories) {
                foreach (explode(',', $categories) as $category) {
                    $query->where('product_categories.id', '=', $category);
                }
            });
        });
    }

    public function scopeActive($query, $active)
    {
        return $query->when($active !== null, function ($query) use ($active) {
            $query->where('activated', '=', boolval($active));
        });
    }

    public function scopeBestCollection($query, $bestCollection)
    {
        return $query->when($bestCollection !== null, function ($query) use ($bestCollection) {
            $query->where('best_collection', '=', $bestCollection == '1' || $bestCollection == 1 || $bestCollection == true || $bestCollection == 'true' ? '1' : '0');
        })->orderByDesc('id');
    }

    public function scopeNewlyAdded($query, $newlyAdded)
    {
        return $query->when($newlyAdded !== null, function ($query) use ($newlyAdded) {
            $query->where('newly_added', '=', $newlyAdded == '1' || $newlyAdded == 1 || $newlyAdded == true || $newlyAdded == 'true' ? '1' : '0');
        })->orderByDesc('id');;
    }

    public function scopeFlashDeals($query, $flashDeal)
    {
        return $query->when($flashDeal !== null, function ($query) use ($flashDeal) {
            $query->where('flash_deal', '=', $flashDeal == '1' || $flashDeal == 1 || $flashDeal == true || $flashDeal == 'true' ? '1' : '0');
        });
    }

    public function scopeSelected($query, $selected)
    {
        return $query->when($selected !== null, function ($query) use ($selected) {
            $query->where('selected', '=', $selected == '1' || $selected == 1 || $selected == true || $selected == 'true' ? '1' : '0');
        });
    }

    public function scopeLatestProducts($query, $latest)
    {
        return $query->when($latest !== null && $latest == true, function ($query) use ($latest) {
            $query->latest()->take(10);
        });
    }

//
    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    public function seller(): BelongsTo
    {
        return $this->belongsTo(Seller::class);
    }

    public function associations(): HasMany
    {
        return $this->hasMany(ProductAssociation::class, 'product_id', 'id');
    }

    public function colors(): HasMany
    {
        return $this->hasMany(Color::class);
    }

    public function sizes(): HasMany
    {
        return $this->hasMany(Size::class);
    }

    public function outOfStock(): BelongsTo
    {
        return $this->belongsTo(OutOfStock::class);
    }

    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(ProductCategory::class, 'category_product', 'product_id', 'category_id');
    }

    public function attributes(): BelongsToMany
    {
        return $this->belongsToMany(Attribute::class, 'product_attributes')
            ->withPivot('value_en', 'value_ar')
            ->as('attributeValue');
    }

    public function availableDeliveryOptions(): BelongsToMany
    {
        return $this->belongsToMany(DeliveryOption::class, 'delivery_option_product', 'product_id', 'delivery_option_id');
    }

    public function availablePayments(): BelongsToMany
    {
        return $this->belongsToMany(Payment::class, 'payment_product', 'product_id', 'payment_id');
    }

    public function reviews(): HasMany
    {
        return $this->hasMany(Review::class);
    }
}
