<?php

namespace App\Models;

use App\Http\Traits\HasMediaImage;
use App\Http\Traits\HasSearch;
use App\Http\Traits\HasUniqueSoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Khsing\World\Models\City;
use Laravel\Passport\HasApiTokens;
use Spatie\MediaLibrary\HasMedia;


class User extends Authenticatable implements HasMedia
{
    use HasFactory, Notifiable, HasApiTokens, SoftDeletes, HasSearch, HasMediaImage, HasUniqueSoftDeletes;

    protected $searchable = [
        'first_name',
        'last_name',
        'email',
        'phone'
    ];
    protected $dateFormat = 'Y-m-d H:i:s';

    protected $guarded = ['id'];

    protected $hidden = [
        'password',
        'remember_token',
        'media',
        'email_verified_at',
        'last_ip',
        'is_logged_in',
        'access_failed_count',
        'last_login_date'
    ];
    protected $dates = [
        'email_verified_at',
        'last_login_date'
    ];
    protected $casts = [
        'activated' => 'boolean',
        'verified' => 'boolean',
        'is_logged_in' => 'boolean',
        'accepted' => 'boolean',
        'image' => 'array'
    ];
    protected $appends = [
        'full_name',
        'super_admin',
        'admin',
        'is_seller',
        'member',
        'delivery'
    ];

    protected static function booted()
    {
        static::created(function ($item) {
            AppPreference::query()->create([
                'user_id' => $item->id
            ]);
        });
    }

    public function getFullNameAttribute(): string
    {
        return "$this->first_name $this->last_name";
    }

    public function getSuperAdminAttribute(): bool
    {
        return $this->role === "Super Admin";
    }

    public function getAdminAttribute(): bool
    {
        return $this->role === "Admin";
    }

    public function getIsSellerAttribute(): bool
    {
        return $this->role === "Seller";
    }

    public function getMemberAttribute(): bool
    {
        return $this->role === "Member";
    }

    public function getDeliveryAttribute(): bool
    {
        return $this->role === "Delivery";
    }

    public function setPasswordAttribute($value)
    {
        if ($value === null || $value == '')
            return;
        $this->attributes['password'] = bcrypt($value);
    }

    public function scopePhoneFilter($query, $phone)
    {
        return $query->when($phone !== null, function ($query) use ($phone) {
            $query->where('phone', 'LIKE', "%$phone%");
        });
    }

    public function scopePhoneOrEmail($query, $phone, $email)
    {
        return $query->when($phone !== null, function ($query) use ($phone) {
            $query->where('phone', '=', $phone);
        })->when($email !== null, function ($query) use ($email) {
            $query->where('email', '=', $email);
        });
    }

    public function scopeRoleFilter($query, $role)
    {
        return $query->when($role !== null, function ($query) use ($role) {
            $query->where('role', '=', $role);
        });
    }

    public function scopeActive($query, $active)
    {
        return $query->when($active !== null, function ($query) use ($active) {
            $query->where('activated', '=', boolval($active));
        });
    }

    public function scopeDelivery($query)
    {
        $query->where('role', '=', 'Delivery');
    }

    public function scopeSeller($query)
    {
        $query->where('role', '=', 'Seller');
    }

    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class);
    }

    public function city(): BelongsTo
    {
        return $this->belongsTo(City::class);
    }

    public function appPreference(): HasOne
    {
        return $this->hasOne(AppPreference::class);
    }

    public function seller(): HasOne
    {
        return $this->hasOne(Seller::class);
    }

    public function favoriteProducts(): BelongsToMany
    {
        return $this->belongsToMany(Product::class, 'product_user');
    }
}
