<?php


namespace App\Services\Checkout;


use App\Models\Address;
use App\Models\Coupon;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use App\Services\Geo\GeoService;
use App\Services\HashService;
use Carbon\Carbon;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Throwable;

class CheckoutServiceImp implements CheckoutService
{
    private $geoService;
    private $cryptoService;

    public function __construct(GeoService $geoService, HashService $hashService)
    {
        $this->geoService = $geoService;
        $this->cryptoService = $hashService;
    }

    public function checkout(array $data, bool $asDraft)
    {
        DB::beginTransaction();

        $productsIDs = [];
        foreach ($data['items'] as $item) {
            array_push($productsIDs, $item['product_id']);
        }
        $products = Product::query()
            ->with([
                'categories'
                , 'brand' => function ($query) {
                    $query->withTrashed();
                }
                , 'seller' => function ($query) {
                    $query->withTrashed();
                }
                , 'outOfStock' => function ($query) {
                    $query->withTrashed();
                }
                , 'availableDeliveryOptions' => function ($query) use ($data) {
                    $query->withTrashed();
                }
                , 'availablePayments' => function ($query) use ($data) {
                    $query->withTrashed();
                }
                , 'colors' => function ($query) {
                    $query->withTrashed();
                }
                , 'sizes' => function ($query) {
                    $query->withTrashed();
                }
                , 'attributes' => function ($query) {
                    $query->withTrashed();
                },
                'attributes.group' => function ($query) {
                    $query->withTrashed();
                }
            ])
            ->whereIn('id', $productsIDs)
            ->where('activated', '=', true)
            ->get()
            ->groupBy('id');
        $user = Auth::user();
        $userAddress = null;
        if ($data['address_id'] == null) {
            $deliverable = false;
        } else {
            $userAddress = Address::query()->where('user_id', '=', $user->id)->findOrFail($data['address_id']);
            $deliverable = $this->deliverable($userAddress);
        }
        $orderSubTotal = 0;
        $orderDiscount = 0;
        $taxTotal = 0;
        $orderTotalQuantity = 0;
        $selectedDeliveryOption = null;
        $selectedPayment = null;
        $orderItems = [];
        $orderTime = Carbon::now($user->timezone)->toDateTimeString();
        foreach ($data['items'] as $item) {
            $productCollection = $products->get($item['product_id']);
            $product = $productCollection != null ? $productCollection->first() : null;
            if ($product == null) {
                throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_item')], 400));
            }
            $localizedProductName = $user->locale == 'en' ? $product->name_en : $product->name_ar;
            if ($product->min_quantity > $item['quantity'] || ($product->max_quantity !== null && $product->max_quantity < $item['quantity'])) {
                if ($product->max_quantity !== null) {
                    throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_quantity', [
                        'itemName' => $localizedProductName,
                        'min_quantity' => $product->min_quantity,
                        'max_quantity' => $product->max_quantity,
                    ])], 400));
                } else {
                    throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_quantity_minimum', [
                        'itemName' => $localizedProductName,
                        'min_quantity' => $product->min_quantity
                    ])], 400));
                }
            }
            if ($product->quantity < $item['quantity']) {
                throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_item')], 400));
            }
            if ($product->colors->filter(function ($item) {
                    return $item->deleted_at === null;
                })->count() > 0 && $item['color_id'] == null) {
                throw new HttpResponseException(response()->json(['error' => __('common.error_color_is_required', ['itemName' => $localizedProductName])], 400));
            }
            if ($product->sizes->filter(function ($item) {
                    return $item->deleted_at === null;
                })->count() > 0 && $item['size_id'] == null) {
                throw new HttpResponseException(response()->json(['error' => __('common.error_size_is_required', ['itemName' => $localizedProductName])], 400));
            }
            $selectedDeliveryOption = $product->availableDeliveryOptions->first(function ($item) use ($data) {
                return $item->id == $data['delivery_option_id'];
            });
            if ($deliverable) {
                if ($selectedDeliveryOption == null) {
                    throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_delivery_option', ['itemName' => $localizedProductName])], 400));
                }
            }

            $selectedPayment = $product->availablePayments->first(function ($item) use ($data) {
                return $item->id == $data['payment_id'];
            });
            if ($selectedPayment == null) {
                throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_payment_null', ['itemName' => $localizedProductName])], 400));
            }
            /* if (!$deliverable && $selectedDeliveryOption->type != 'pickup') {
                 throw new HttpResponseException(response()->json(['error' => __('common.error_not_deliverable', ['itemName' => $localizedProductName])], 400));
             }*/
            /*if ($selectedPayment->type != 'offline' && $data['billing_address'] == null) {
                throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_payment_requires_billing', ['itemName' => $localizedProductName])], 400));
            }*/
            if ($selectedPayment->type == 'offline' && $product->subtract_stock) {
                Product::query()->where('id', '=', $item['product_id'])->decrement('quantity', $item['quantity']);
            }
            $itemPrice = $product->price;
            $discount = 0;
            $specialOfferTotal = 0;
            if ($product->active_special_offer && ($product->special_offer['price'] !== null || $product->special_offer['percentage'] !== null)) {
                if ($product->special_offer['price'] !== null) {
                    $specialOfferTotal = $itemPrice = $product->special_offer['price'];
                } else {
                    $specialOfferTotal = $itemPrice = $product->price - ($product->special_offer['percentage'] / 100 * $product->price);
                }
            }
            if ($product->active_discount) {
                if ($item['quantity'] >= $product->discount['quantity']) {
                    if ($product->discount['price'] !== null) {
                        $discount = $product->discount['price'] / $item['quantity'];
                        $itemPrice -= ($product->discount['price'] / $item['quantity']);
                    } else {
                        $discount = ($product->discount['price'] / $item['quantity']);
                        $itemPrice -= (($product->discount['percentage'] / 100 * $itemPrice) / $item['quantity']);
                    }
                }
            }
            if ($itemPrice < 0) {
                $itemPrice = $product->price;
            }
            $tax = 0;
            $selectedColor = $product->colors->first(function ($color) use ($item) {
                return $color->id == $item['color_id'];
            });
            $selectedSize = $product->sizes->first(function ($size) use ($item) {
                return $size->id == $item['size_id'];
            });
            array_push($orderItems, [
                'date' => $orderTime,
                'unit_name_en' => $product->name_en,
                'unit_name_ar' => $product->name_ar,
                'unit_dimensions' => json_encode($product->dimensions),
                'unit_weight' => $product->weight,
                'unit_price' => $product->price,
                'unit_discount' => json_encode($product->discount),
                'unit_special_offer' => json_encode($product->special_offer),
                'unit_image' => json_encode($product->image),
                'unit_color' => $selectedColor === null ? null : json_encode($selectedColor),
                'unit_size' => $selectedSize === null ? null : json_encode($selectedSize),
                'brand_name_en' => $product->brand?->name_en,
                'brand_name_ar' => $product->brand?->name_ar,
                'tax' => $tax,
                'subtotal' => $itemPrice * $item['quantity'],
                'total' => ($itemPrice * $item['quantity']) + $tax,
                'discount_total' => $discount,
                'special_offer_total' => $specialOfferTotal,
                'quantity' => $item['quantity'],
                'subtracted' => $product->subtract_stock ? $item['quantity'] : 0,
                'notes' => $item['notes'],
                'product_id' => $item['product_id'],
                'seller_id' => $product->seller_id,
            ]);
            $orderSubTotal += ($itemPrice * $item['quantity']);
            $orderDiscount += $discount;
            $taxTotal += $tax;
            $orderTotalQuantity += $item['quantity'];
        }
        $postProcess = Str::lower($selectedPayment->type) != 'offline' || $asDraft;
        $deliveryTotal = $selectedDeliveryOption == null ? 0 : ($selectedDeliveryOption->cost);
        $billingAddress = null;
        if (array_key_exists('billing_address',$data)&&$data['billing_address'] !== null) {
            $billingAddress = Address::query()->create(array_merge(Arr::only($data['billing_address'][0], [
                'first_name',
                'last_name',
                'company',
                'street_address',
                'province',
                'city',
                'floor',
                'region',
                'postal_code',
                'phone',
                'district',
                'country_id',
                'boulevard',
                'house',
                'apartment',
            ]), [
                'longitude' => $userAddress != null ? $userAddress->longitude : null,
                'latitude' => $userAddress != null ? $userAddress->latitude : null,
            ]));
        }
        $shippingAddress = null;
        if (array_key_exists('shipment_address',$data)&&$data['shipment_address'] !== null) {
            $shippingAddress = Address::query()->create(array_merge(Arr::only($data['shipment_address'][0], [
                'first_name',
                'last_name',
                'company',
                'street_address',
                'province',
                'city',
                'floor',
                'region',
                'postal_code',
                'phone',
                'district',
                'country_id',
                'boulevard',
                'house',
                'apartment',
            ]), [
                'longitude' => $userAddress != null ? $userAddress->longitude : null,
                'latitude' => $userAddress != null ? $userAddress->latitude : null,
            ]));
        }
        $couponTotal = 0;
        if ($data['coupon_id'] !== null) {
            $coupon = Coupon::query()->where('id', '=', $data['coupon_id'])->where('user_id', '=', $user->id)->first();
            if ($coupon == null) {
                throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_coupon')], 400));
            }
            if (!$coupon->valid_now) {
                throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_coupon')], 400));
            }
            if ($coupon->on_delivery_cost) {
                $deliveryTotal = 0;
            }
            if ($coupon->amount == null) {
                $couponTotal = ($orderSubTotal * ($coupon->percentage / 100));
            } else {
                $couponTotal = $coupon->amount;
            }
            if (!$postProcess) {
                $coupon->update(['used' => true]);
            }
        }
        $lastOrder = Order::query()->orderByDesc('date')->first();
        $lastNumber = ($lastOrder == null ? "0" : $lastOrder->id);
        /*$orderNumber = '';
        for ($i = 0; $i < max(0, 10 - strlen(strval($lastNumber))); $i++) {
            $orderNumber .= rand(0, 9);
        }
        $deliveryCode = '';
        for ($i = 0; $i < 6; $i++) {
            $orderNumber .= rand(0, 9);
        }*/
        $order = Order::query()->create([
            'date' => $orderTime,
            'number' => '#' . $this->cryptoService->numericHash($lastNumber, strlen($lastNumber) < 10 ? 10 : strlen($lastNumber)),
            'delivery_code' => $this->cryptoService->numericHash($lastNumber, 6),
            'state' => 'new',
            'total' => $orderSubTotal + $deliveryTotal + $taxTotal - $couponTotal,
            'subtotal' => $orderSubTotal,
            'currency' => request()->query('currency', 'USD'),
            'total_quantity' => $orderTotalQuantity,
            'notes' => $data['notes'],
            'discount_total' => $orderDiscount,
            'coupon_total' => $couponTotal,
            'delivery_method' => $selectedDeliveryOption == null ? 'delivery' : ($selectedDeliveryOption->type),
            'delivery_total' => $deliveryTotal,
            'deliverable' => $deliverable,
            'payment_state' => 'awaiting',
            'shipping_state' => $deliverable ? null : 'ready',
            'user_id' => $user->id,
            'delivery_id' => null,
            'billing_address_id' => $billingAddress == null ? null : $billingAddress->id,
            'shipping_address_id' => $shippingAddress == null ? null : $shippingAddress->id,
            'address_id' => $userAddress == null ? null : $userAddress->id,
            'payment_id' => $data['payment_id'],
            'delivery_option_id' => $data['delivery_option_id'],
            'coupon_id' => $data['coupon_id'],
            'deleted_at' => null,
            'post_process' => $postProcess,
            'payment_platform' => array_key_exists('payment_platform', $data) ? $data['payment_platform'] : null,
        ]);
        if ($postProcess) {
            $postUrl = "https://api.talabatexpress.com/api/order/postCallback?order_id=$order->id";
            $order->update(['post_url' => $postUrl]);
        }
        $finalOrderItems = [];
        foreach ($orderItems as $orderItem) {
            $orderItem['order_id'] = $order->id;
            array_push($finalOrderItems, $orderItem);
        }
        OrderItem::query()->insert($finalOrderItems);
        DB::commit();
        return $order->load([
            'user'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress.city'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
        ]);
    }

    public function deliverable($address): bool
    {
        $country = $this->geoService->googleGeoCode($address);
        if ($country == 'unknown') {
            throw new HttpResponseException(response()->json(['error' => __('common.error_unable_to_fetch_location')], 400));
        }
        if (Str::contains('kuwait', Str::lower($country))) {
            return true;
        }
        return false;
    }

    public function postProcess(Request $request): array
    {
        try {
            DB::beginTransaction();
            $requestBody = json_encode($request->all());
            $requestHeaders = json_encode($request->header());
            $secretKey = config('app.tap_secret');
            $toBeHashedString = 'x_id' . $request->input('id') .
                'x_amount' . $request->input('amount') .
                'x_currency' . $request->input('currency') .
                'x_gateway_reference' . $request->input('reference.gateway') .
                'x_payment_reference' . $request->input('reference.payment') .
                'x_status' . $request->input('status') .
                'x_created' . $request->input('transaction.created');

            if (Str::lower($request->input('status')) !== 'captured') {
                Log::channel('order_post')->info("
                ############################ Order PostUrl Exception ##############\n
                The status is not CAPTURED\n
                Status: {$request->input('status')}\n
                Request All:\n
                $requestBody
                \n
                --------------------------------------------\n
                Request Headers:\n
                $requestHeaders
                \n
                --------------------------------------------\n
                ");
                return [
                    'order' => null,
                    'status' => 'invalid post status'
                ];
            }

//            $generatedHashString = hash_hmac('sha256', $toBeHashedString, $secretKey);
//            $postedHashString = $request->header('hashstring');
//            if ($generatedHashString !== $postedHashString) {
//                Log::channel('order_post')->info("
//                ############################ Order PostUrl Exception ##############\n
//                Generated HasString does not match the posted Hash String\n
//                Generated HasString:   $generatedHashString\n
//                Posted HasString:   $postedHashString\n\n
//                Request All:\n
//                $requestBody
//                \n
//                --------------------------------------------\n
//                Request Headers:\n
//                $requestHeaders
//                \n
//                --------------------------------------------\n
//                ");
//            }

            $orderId = $request->input('order_id');
            $order = Order::withTrashed(true)->find($orderId);
            if ($order == null) {
                Log::channel('order_post')->info("
                ############################ Order PostUrl Exception ##############\n
                Order not found\n
                Order:\n
                ID: {$request->input('order_id')}\n
                Request All:\n
                $requestBody
                \n
                --------------------------------------------\n
                Request Headers:\n
                $requestHeaders
                \n
                --------------------------------------------\n
                ");
                return [
                    'order' => null,
                    'status' => 'order not found'
                ];
            }
//            if ($order->deleted_at == null) {
//                Log::channel('order_post')->info("
//                ############################ Order PostUrl Exception ##############\n
//                Order already processed\n
//                Order:\n
//                ID: {$request->input('order_id')}\n
//                Request All:\n
//                $requestBody
//                \n
//                --------------------------------------------\n
//                Request Headers:\n
//                $requestHeaders
//                \n
//                --------------------------------------------\n
//                ");
//                return [
//                    'order' => $order,
//                    'status' => 'already processed'
//                ];
//            }
//            $talabatQueryHash = $request->query('talabat_hash');
//            if ($talabatQueryHash == null) {
//                Log::channel('order_post')->info("
//                ############################ Order Talabat Hash Exception ##############\n
//                Invalid Talabat Hash
//                Request All:\n
//                $requestBody
//                \n
//                --------------------------------------------\n
//                Request Headers:\n
//                $requestHeaders
//                \n
//                --------------------------------------------\n
//                ");
//                return [
//                    'order' => $order,
//                    'status' => 'null hash string'
//                ];
//            }
//            $talabatGeneratedHashString = hash_hmac('sha256', "{$order->user->id}-{$order->id}", config("app.talabat_secret"));
//            $key = config("app.talabat_secret");
//            if ($talabatQueryHash !== $talabatGeneratedHashString) {
//                Log::channel('order_post')->info("
//                ############################ Order Talabat Hash Exception ##############\n
//                Generated HasString does not match the posted Hash String\n
//                Generated HasString:   $talabatGeneratedHashString\n
//                Posted HasString:   $talabatQueryHash\n\n
//                Hashed: {$order->user->id}-{$order->id}\n
//                Key: $key
//                Request All:\n
//                $requestBody
//                \n
//                --------------------------------------------\n
//                Request Headers:\n
//                $requestHeaders
//                \n
//                --------------------------------------------\n
//                ");
//                return [
//                    'order' => $order,
//                    'status' => 'invalid hash string'
//                ];
//            }
            $order->update([
                'payment_state' => 'paid',
                'deleted_at' => null
            ]);
            foreach ($order->orderItems as $orderItem) {
                if ($orderItem->product->subtract_stock) {
                    Product::query()->where('id', '=', $orderItem->product->id)->decrement('quantity', $orderItem->quantity);
                }
            }
            if ($order->coupon !== null) {
                $order->coupon()->update(['used' => true]);
            }
            DB::commit();
//            Log::channel('order_post')->info("
//                ############################ Order PostUrl Successfully Processed ##############\n
//                Talabat Generated HasString:   $talabatGeneratedHashString\n
//                Talabat Posted HasString:   $talabatQueryHash\n\n
//                Generated HasString:   $generatedHashString\n
//                Posted HasString:   $postedHashString\n\n
//                Request All:\n
//                $requestBody
//                \n
//                --------------------------------------------\n
//                Request Headers:\n
//                $requestHeaders
//                \n
//                --------------------------------------------\n
//                Order:\n
//                ID: {$request->input('order_id')}\n
//                ");
            return [
                'order' => $order->load([
                    'user'
                    , 'delivery'
                    , 'coupon'
                    , 'billingAddress'
                    , 'shippingAddress.city'
                    , 'seller'
                    , 'payment'
                    , 'deliveryOption'
                    , 'orderItems.product'
                ]),
                'status' => 'success'
            ];
        } catch (Throwable $exception) {
            DB::rollBack();
            $requestBody = json_encode($request->all());
            $requestHeaders = json_encode($request->header());
            Log::channel('order_post')->info("
                ############################ Order PostUrl Exception ##############\n
                500 Error\n
                {$exception->getMessage()}\n
                {$exception->getLine()}\n
                ----- {$exception->getTraceAsString()} -----\n
                Request All:\n
                $requestBody
                \n
                --------------------------------------------\n
                Request Headers:\n
                $requestHeaders
                \n
                --------------------------------------------\n
                ");
            return [
                'order' => null,
                'status' => 'exception'
            ];
        }
    }
}
